<?php
/*
Title: 🌀 WebSocket Chat Script
Author: K0NxT3D
Description: Simple Web Sockets Client - Server Chat Script.
File: server.php
URL: http://k0nxt3d.seaverns.com/
Use:
    On Server Host Run : php server.php
    On Client Host Open client.php.
        Make sure The $host matches in client.php:
        Line 107: const ws = new WebSocket('ws://127.0.0.1:6969');
*/

// Set Host Details
$host = '127.0.0.1'; // Host to bind to
$port = 6969; // Port to bind to

set_time_limit(0); // Prevent Timeouts

// Create TCP/IP Socket
$socket = socket_create(AF_INET, SOCK_STREAM, SOL_TCP);
if ($socket === false) {
    die("Could not create socket: " . socket_strerror(socket_last_error()) . "\n");
}

// Bind The Socket
if (!socket_bind($socket, $host, $port)) {
    die("Could not bind to socket: " . socket_strerror(socket_last_error($socket)) . "\n");
}

// Listen For Connections
if (!socket_listen($socket)) {
    die("Could not set up socket listener: " . socket_strerror(socket_last_error($socket)) . "\n");
}

$clients = []; // Array To Hold Connected Clients

// Function To Perform WebSocket Handshake
function performHandshake($clientSocket, $headers) {
    $key = '';
    if (preg_match('/Sec-WebSocket-Key: (.*)\r\n/', $headers, $matches)) {
        $key = $matches[1];
    }
    $acceptKey = base64_encode(pack('H*', sha1($key . '258EAFA5-E914-47DA-95CA-C5AB0DC85B11')));
    $headers = "HTTP/1.1 101 Switching Protocols\r\n" .
        "Upgrade: websocket\r\n" .
        "Connection: Upgrade\r\n" .
        "Sec-WebSocket-Accept: $acceptKey\r\n\r\n";
    socket_write($clientSocket, $headers, strlen($headers));
}

// Function To Decode WebSocket Messages
function decodeMessage($data) {
    if (empty($data)) return null; // Return null if data is empty

    $length = ord($data[1]) & 127;
    $decoded = '';

    if ($length === 126) {
        $length = unpack('n', substr($data, 2, 2))[1];
        $mask = substr($data, 4, 4);
        $message = substr($data, 8);
    } elseif ($length === 127) {
        $length = unpack('P', substr($data, 2, 8))[1];
        $mask = substr($data, 10, 4);
        $message = substr($data, 14);
    } else {
        $mask = substr($data, 2, 4);
        $message = substr($data, 6);
    }

    for ($i = 0; $i < $length; $i++) {
        $decoded .= $message[$i] ^ $mask[$i % 4];
    }

    return $decoded; // Return the decoded message
}

// Function To Send A Message To All Connected Clients
function sendMessageToAll($message, $clients) {
    $encodedMessage = "\x81" . chr(strlen($message)) . $message;
    foreach ($clients as $client) {
        socket_write($client, $encodedMessage, strlen($encodedMessage));
    }
}

while (true) {
    $read = $clients;
    // Remove The Server Socket From The Read Array.
    $read[] = $socket; // Add The Server Socket To The Read Array.

    // Check For Activity On The Sockets.
    if (socket_select($read, $null, $null, 0) < 1) {
        continue; // No Activity
    }

    // Check For New Connections.
    if (in_array($socket, $read)) {
        $newClient = socket_accept($socket);
        $headers = socket_read($newClient, 1024);
        performHandshake($newClient, $headers);
        $clients[] = $newClient; // Add New Client To The List.
        echo "New Client Connected.\n";
        // Remove The Server Socket From The Read Array For The Next Iteration.
        unset($read[array_search($socket, $read)]);
    }

    // Loop Through The Clients To Check For Messages.
    foreach ($clients as $key => $client) {
        if (in_array($client, $read)) {
            $input = socket_read($client, 1024);

            if ($input === false) {
                unset($clients[$key]); // Remove Client If Disconnected.
                echo "Client Disconnected.\n";
                continue;
            }

            // Decode The Message And Broadcast It.
            $messageData = decodeMessage($input);
            if ($messageData !== null) {
                sendMessageToAll($messageData, $clients);
            }
        }
    }
}

// Close The Socket When Finished.
socket_close($socket);
?>

